function color_img = map2color(img,varargin)
%This little function takes a grayscale image and maps the image based on
%intensity to a color map.
%Syntax: color_img = map2color(img,'colormap','','depth',[]);
%Input: img = the intensity image in either uint8 or double, if double the
%       value of the image must be between 0 and 1.
%       colormap = the colormap you want to use.  Default = fir2
%       depth = the color depth.  Default = 256
%Output:  color_img = the color image as a 3D matrix

%parse the input
[map_type,depth] = parse(varargin);

%initialize basic parameters
[a,b] = size(img);      %size of the image
color_img = zeros(a,b,3);    %initialize color matrix

%create the colormap array
cmap = feval(map_type,depth);

%map the map onto the intensity image
if isa(img,'double')         %double image
    if max(max(img))>1||min(min(img))<0     %double image must be between 0 and 1
        error(['Error: values in a double image must be between 0 and 1']);
    else
        img = round(img*(depth-1));     %transform the image into an index array for the color map.
        %note: matlab uint8 image arrays are 0-255, but color map indexes
        %are 1-256
        img = img+1;        %shift the image array up one to produce a usable index. 0 is not a valid value
        for i = 1:a
            for j = 1:b
                color_img(i,j,:) = cmap(img(i,j),:);
            end
        end
    end
elseif isa(img,'uint8')    %uint8 image
    img = img+1;        %see above comment
    for i = 1:a     %step through rows
        for j = 1:b     %step through columns
            color_img(i,j,:) = cmap(img(i,j),:);    %map the value
        end
    end
else
    error(['Error: Your image type is not recognized.  Currently only uint8 and double images are supported']);
end


%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [map_type,depth] = parse(input)
%set defaults
map_type = 'fir2';
depth = 256;

%Parse the input
if ~isempty(input)
    a = size(input,2);
    for i = 1:2:a
        value = input{1,i+1};
        switch input{1,i}
            case 'depth'
                if isnumeric(value)
                    depth = value;
                else
                    error(['Error: You did not enter the proper value for the depth property']);
                end
            case 'colormap'
                if ischar(value)
                    map_type = value;
                else
                    error(['Error: You did not enter teh proper value for the colormap property']);
                end
        end
    end
end